<?php
class CronJob extends CronnerAppModel {

	/**
	 * Get global common options
	 *
	 * @return array
	 */
	public function getCommonOptions()
	{
		return array(
			"* * * * *" => "Once Per Minute (* * * * *)",
			"*/5 * * * *" => "Once Per Five Minutes (*/5 * * * *)",
			"0,30 * * * *" => "Twice Per Hour (0,30 * * * *)",
			"0 * * * *" => "Once Per Hour (0 * * * *)",
			"0 0,12 * * *" => "Twice Per Day (0 0,12 * * *)",
			"0 0 * * *" => "Once Per Day (0 0 * * *)",
			"0 0 * * 0" => "Once Per Week (0 0 * * 0)",
			"0 0 1,15 * *" => "On the 1st and 15th of the Month (0 0 1,15 * *)",
			"0 0 1 * *" => "Once Per Month (0 0 1 * *)",
			"0 0 1 1 *" => "Once Per Year (0 0 1 1 *)"
		);
	}

	/**
	 * Get available options for minute select
	 *
	 * @return array
	 */
	public function getMinuteOptions()
	{
		return array(
			"*" => "Once Per Minute (*)",
			"*/2" => "Once Per Two Minutes (*/2)",
			"*/5" => "Once Per Five Minutes (*/5)",
			"*/10" => "Once Per Ten Minutes (*/10)",
			"*/15" => "Once Per Fifteen Minutes (*/15)",
			"0,30" => "Once Per Thirty Minutes (0,30)",
			"--" => "-- Minutes --",
			"0" => ":00 (At the beginning of the hour.) (0)",
			"1" => ":01 (1)",
			"2" => ":02 (2)",
			"3" => ":03 (3)",
			"4" => ":04 (4)",
			"5" => ":05 (5)",
			"6" => ":06 (6)",
			"7" => ":07 (7)",
			"8" => ":08 (8)",
			"9" => ":09 (9)",
			"10" => ":10 (10)",
			"11" => ":11 (11)",
			"12" => ":12 (12)",
			"13" => ":13 (13)",
			"14" => ":14 (14)",
			"15" => ":15 (At one quarter past the hour.) (15)",
			"16" => ":16 (16)",
			"17" => ":17 (17)",
			"18" => ":18 (18)",
			"19" => ":19 (19)",
			"20" => ":20 (20)",
			"21" => ":21 (21)",
			"22" => ":22 (22)",
			"23" => ":23 (23)",
			"24" => ":24 (24)",
			"25" => ":25 (25)",
			"26" => ":26 (26)",
			"27" => ":27 (27)",
			"28" => ":28 (28)",
			"29" => ":29 (29)",
			"30" => ":30 (At half past the hour.) (30)",
			"31" => ":31 (31)",
			"32" => ":32 (32)",
			"33" => ":33 (33)",
			"34" => ":34 (34)",
			"35" => ":35 (35)",
			"36" => ":36 (36)",
			"37" => ":37 (37)",
			"38" => ":38 (38)",
			"39" => ":39 (39)",
			"40" => ":40 (40)",
			"41" => ":41 (41)",
			"42" => ":42 (42)",
			"43" => ":43 (43)",
			"44" => ":44 (44)",
			"45" => ":45 (At one quarter until the hour.) (45)",
			"46" => ":46 (46)",
			"47" => ":47 (47)",
			"48" => ":48 (48)",
			"49" => ":49 (49)",
			"50" => ":50 (50)",
			"51" => ":51 (51)",
			"52" => ":52 (52)",
			"53" => ":53 (53)",
			"54" => ":54 (54)",
			"55" => ":55 (55)",
			"56" => ":56 (56)",
			"57" => ":57 (57)",
			"58" => ":58 (58)",
			"59" => ":59 (59)"
		);
	}

	/**
	 * Get available options for hour select
	 *
	 * @return array
	 */
	public function getHourOptions()
	{
		return array(
			"*" => "Every Hour (*)",
			"*/2" => "Every Other Hour (*/2)",
			"*/3" => "Every Third Hour (*/3)",
			"*/4" => "Every Fourth Hour (*/4)",
			"*/6" => "Every Sixth Hour (*/6)",
			"0,12" => "Every Twelve Hours (0,12)",
			"--" => "-- Hours --",
			"0" => "12:00 a.m. Midnight (0)",
			"1" => "1:00 a.m. (1)",
			"2" => "2:00 a.m. (2)",
			"3" => "3:00 a.m. (3)",
			"4" => "4:00 a.m. (4)",
			"5" => "5:00 a.m. (5)",
			"6" => "6:00 a.m. (6)",
			"7" => "7:00 a.m. (7)",
			"8" => "8:00 a.m. (8)",
			"9" => "9:00 a.m. (9)",
			"10" => "10:00 a.m. (10)",
			"11" => "11:00 a.m. (11)",
			"12" => "12:00 p.m. Noon (12)",
			"13" => "1:00 p.m. (13)",
			"14" => "2:00 p.m. (14)",
			"15" => "3:00 p.m. (15)",
			"16" => "4:00 p.m. (16)",
			"17" => "5:00 p.m. (17)",
			"18" => "6:00 p.m. (18)",
			"19" => "7:00 p.m. (19)",
			"20" => "8:00 p.m. (20)",
			"21" => "9:00 p.m. (21)",
			"22" => "10:00 p.m. (22)",
			"23" => "11:00 p.m. (23)"
		);
	}

	/**
	 * Get available options for day select
	 *
	 * @return array
	 */
	public function getDayOptions()
	{
		return array(
			"*" => "Every Day (*)",
			"*/2" => "Every Other Day (*/2)",
			"1,15" => "On the 1st and 15th of the Month (1,15)",
			"--" => "-- Days --",
			"1" => "1st (1)",
			"2" => "2nd (2)",
			"3" => "3rd (3)",
			"4" => "4th (4)",
			"5" => "5th (5)",
			"6" => "6th (6)",
			"7" => "7th (7)",
			"8" => "8th (8)",
			"9" => "9th (9)",
			"10" => "10th (10)",
			"11" => "11th (11)",
			"12" => "12th (12)",
			"13" => "13th (13)",
			"14" => "14th (14)",
			"15" => "15th (15)",
			"16" => "16th (16)",
			"17" => "17th (17)",
			"18" => "18th (18)",
			"19" => "19th (19)",
			"20" => "20th (20)",
			"21" => "21st (21)",
			"22" => "22nd (22)",
			"23" => "23rd (23)",
			"24" => "24th (24)",
			"25" => "25th (25)",
			"26" => "26th (26)",
			"27" => "27th (27)",
			"28" => "28th (28)",
			"29" => "29th (29)",
			"30" => "30th (30)",
			"31" => "31st (31)"
		);
	}

	/**
	 * Get available options for month select
	 *
	 * @return array
	 */
	public function getMonthOptions()
	{
		return array(
			"*" => "Every Month (*)",
			"*/2" => "Every Other Month (*/2)",
			"*/4" => "Every Third Month (*/4)",
			"1,7" => "Every Six Months (1,7)",
			"--" => "-- Months --",
			"1" => "January (1)",
			"2" => "February (2)",
			"3" => "March (3)",
			"4" => "April (4)",
			"5" => "May (5)",
			"6" => "June (6)",
			"7" => "July (7)",
			"8" => "August (8)",
			"9" => "September (9)",
			"10" => "October (10)",
			"11" => "November (11)",
			"12" => "December (12)"
		);
	}

	/**
	 * Get available options for weekday select
	 *
	 * @return array
	 */
	public function getWeekdayOptions()
	{
		return array(
			"*" => "Every Day (*)",
			"1-5" => "Every Weekday (1-5)",
			"0,6" => "Every Weekend Day (6,0)",
			"1,3,5" => "Every Monday, Wednesday, and Friday (1,3,5)",
			"2,4" => "Every Tuesday and Thursday (2,4)",
			"--" => "-- Weekdays --",
			"0" => "Sunday (0)",
			"1" => "Monday (1)",
			"2" => "Tuesday (2)",
			"3" => "Wednesday (3)",
			"4" => "Thursday (4)",
			"5" => "Friday (5)",
			"6" => "Saturday (6)"
		);
	}


	/**
	 * Initialize validation
	 *
	 * @return array
	 */
	public function initValidation()
	{
		return array(
			'name' => array(
				'notBlank' => array(
					'rule' => array('notBlank'),
					'required' => true,
					'message' => __('Please insert job name')
				)
			)
		);
	}

	public function beforeValidate($options = array())
	{
		parent::beforeValidate($options);
		$this->buildExpression();

		// Validate expression
		if(!empty($this->data[$this->alias]['expression'])) {
			try {
				Cron\CronExpression::factory($this->data[$this->alias]['expression']);
			} catch(Exception $e) {
				$this->invalidate('expression', __('Please check all fields. Some of them are invalid!'));
			}
		}
	}

	/**
	 * Build expression
	 */
	private function buildExpression()
	{
		$keys = array(
			'minute',
			'hour',
			'day',
			'month',
			'weekday'
		);

		$values = array();
		foreach($keys as $key) {
			if(!isset($this->data[$this->alias][$key])) {
				return;
			}
			$values[] = $this->data[$this->alias][$key];
		}

		if(isset($this->data[$this->alias]['year'])) {
			$values[] = $this->data[$this->alias]['year'];
		}

		$this->data[$this->alias]['expression'] = join(' ', $values);
	}

	/**
	 * Get last run for provided id
	 *
	 * @param null $id
	 * @return mixed
	 */
	public function getLastRun($id = null)
	{
		return ClassRegistry::init('Cronner.CronJobLog')->getLastRun($this->extractId($id));
	}

	/**
	 * Get next run time for expression
	 *
	 * @param null $expression
	 * @return DateTime
	 */
	public function getNextRunTime($expression = null)
	{
		if(empty($expression)) {
			$expression = $this->data[$this->alias]['expression'];
		}
		$cron = Cron\CronExpression::factory($expression);
		return $cron->getNextRunDate();
	}

	/**
	 * Get currently running for provided id
	 *
	 * @return mixed
	 */
	public function getCurrentlyRunning($id = null)
	{
		return ClassRegistry::init('Cronner.CronJobLog')->getRunning($this->extractId($id));
	}

	/**
	 * Get last runs
	 *
	 * @param null $id
	 * @param int $limit
	 * @return mixed
	 */
	public function getLastRuns($id = null, $limit = 20)
	{
		return ClassRegistry::init('Cronner.CronJobLog')->getLastRuns($this->extractId($id), $limit);
	}


	public function downloadLog($logId)
	{
		return ClassRegistry::init('Cronner.CronJobLog')->downloadLog($logId);
	}

	/**
	 * Get provided or model id
	 *
	 * @param $id
	 * @return array|bool|int|mixed|string
	 */
	private function extractId($id)
	{
		if(empty($id)) {
			return $this->id;
		}
		return $id;
	}

	/**
	 * Get main job command
	 *
	 * @return string
	 */
	public function getMainJobCommand()
	{
		return "*\t*\t*\t*\t*\tphp -f " . ROOT . DS . "app/Console/cake.php cron";
	}

	/**
	 * Check if main job is installed
	 *
	 * @return bool
	 */
	public function mainJobInstalled()
	{
		$output = shell_exec('crontab -l');

		$lines = explode(PHP_EOL, $output);
		$lines = array_filter($lines);

		$validLines = array();

		// Remove invalid lines
		foreach($lines as $line) {
			$line = trim($line);
			if(substr($line, 0, 1) == '#') {
				continue;
			}
			$validLines[] = $line;
		}

		// Match with regular expression
		$jobFound = false;
		$regularExpression = "~^\\*\\s+\\*\\s+\\*\\s+\\*\\s+\\*\\s+php\\s+-f\\s+" . str_replace(DS, '\\/', ROOT . DS . 'app/Console/cake') . ".php\s+cron\s*$~";
		foreach($validLines as $validLine) {
			if(preg_match($regularExpression, $validLine)) {
				$jobFound = true;
			}
		}

		return $jobFound;
	}

}
