<?php
namespace IntellexApps\PHP\UploadFile\Domain\Model;

use IntellexApps\PHP\UploadFile\Domain\Service\FileSystemService;
use IntellexApps\PHP\UploadFile\Domain\Service\PermissionService;
use IntellexApps\PHP\UploadFile\Domain\Service\ServiceContainer;

/**
 * @SWG\Definition(definition="FolderItem", type="object")
 */
class FolderItem extends UploadFileModel
{
    const TYPE_FILE = 'file';
    const TYPE_DIR = 'directory';

    /**
     * @var string
     * @SWG\Property()
     */
    public $type;

    /**
     * @var string
     * @SWG\Property()
     */
    public $name;

    /**
     * @var int
     * @SWG\Property()
     */
    public $size;

    /**
     * @var string
     * @SWG\Property()
     */
    public $base64;

    /**
     * @var string
     * @SWG\Property()
     */
    public $ext;

    /**
     * @var string
     * @SWG\Property()
     */
    public $path;

    /**
     * @var boolean
     * @SWG\Property()
     */
    public $has_children = false;

    /**
     * @var string
     * @SWG\Property()
     */
    public $parent_path;

    /**
     * @var string
     * @SWG\Property()
     */
    public $modified;

    /**
     * @var string
     * @SWG\Property()
     */
    public $permissions = '0777';

    /**
     * @var object
     * @SWG\Property()
     */
    public $parsed_permissions;

    /**
     * @var string
     * @SWG\Property()
     */
    public $client_permissions = 'rw';

    /**
     * @var object
     * @SWG\Property()
     */
    public $parsed_client_permissions;

    /**
     * @var string
     * @SWG\Property()
     */
    public $thumbnail = null;

    /**
     * @var string
     * @SWG\Property()
     */
    public $mimeType = null;

    /**
     * @var string
     * @SWG\Property(
     *     ref="#/definitions/FolderItemMeta"
     * )
     */
    public $fileMeta = null;

    /**
     * Construct folder item
     *
     * FolderItem constructor.
     * @param $path
     */
    public function __construct($path)
    {
        $this->buildFromPath($path);
    }

    /**
     * Build from path
     *
     * @param $path
     */
    private function buildFromPath($path)
    {
        $pathinfo = pathinfo($path);

        $this->type = is_dir($path) ? self::TYPE_DIR : self::TYPE_FILE;
        $this->ext = $pathinfo['extension'];
        $this->name = $pathinfo['basename'];
        $this->path = FileSystemService::relativePath($path);
        $this->parent_path = FileSystemService::relativePath(dirname($path));
        $this->modified = filemtime($path);
        $this->size = filesize($path);
        $this->thumbnail = $this->generateThumbnailUrl($path);


        $this->permissions = FileSystemService::getPermissions($path);
        $this->client_permissions = ServiceContainer::instance()->get('permission')->getPermissionsForPath($path);
        $this->buildParsedClientPermissoins();
        $this->parsePermissions();

        if($this->type == self::TYPE_DIR) {
            $this->has_children = ServiceContainer::instance()->get('upload')->hasChildren(FileSystemService::relativePath($path));
        }

        $this->fileMeta = new FolderItemMeta($path);
    }

    /**
     * Build parsed client permissions
     */
    private function buildParsedClientPermissoins()
    {
        $permissions = [
            'r' => false,
            'w' => false
        ];

        // TODO: Make this look better

        // Read
        if($this->permissions == PermissionService::PERMISSION_READ) {
            $permissions['r'] = true;
        }

        // Write
        if($this->permissions == PermissionService::PERMISSION_WRITE) {
            $permissions['w'] = true;
        }

        // Both
        if($this->permissions == PermissionService::PERMISSION_READ_WRITE) {
            $permissions['r'] = true;
            $permissions['w'] = true;
        }

        // Set parsed permissions
        $this->parsed_client_permissions = $permissions;
    }

    private function parsePermissions()
    {
        $this->parsed_permissions = [
            'world' => PermissionService::getPermissionStringRepresentation($this->permissions[3]),
            'group' => PermissionService::getPermissionStringRepresentation($this->permissions[2]),
            'user' => PermissionService::getPermissionStringRepresentation($this->permissions[1]),
        ];
    }

    private function generateThumbnailUrl($path)
    {
        $mimeType = mime_content_type($path);

        if(substr($mimeType, 0, 5) == 'image') {
            return $this->path;
        }

        return null;
    }
}

/**
 * @SWG\Definition(
 *     definition="FolderItems",
 *     type="array",
 *     @SWG\Items(ref="#/definitions/FolderItem")
 * )
 */