<?php
/**
 * Created by PhpStorm.
 * User: sale
 * Date: 24.10.17.
 * Time: 11.39
 */

namespace Intellex\Generator\Text\HTML;

use Intellex\Generator\Basic\ImageGenerator;
use Intellex\Stopwatch\Exception;

class HTMLParagraphGenerator extends HTMLSentenceGenerator {

	/** @var float $br Chance of generating <br> tag in paragraph in range of 0-1. */
	private $br;

	/** @var int $min Minimum number of sentences in paragraph. */
	private $min;

	/** @var int $max Maximum number of sentences in paragraph. */
	private $max;

	/** @var float $image Chance of generating image in paragraph in range of 0-1. */
	private $image;

	/** @var float $ul Chance of generating <ul> tag in paragraph in range of 0-1. */
	private $ul;

	/** @var float $ol Chance of generating <ol> tag in paragraph in range of 0-1. */
	private $ol;

	/** @var string $imgAPI API Key for pixabay. */
	private $imgAPI;

	/**
	 * HTMLParagraphGenerator constructor.
	 *
	 * @param string $imgAPI API Key for pixabay.
	 * @param float  $br     Chance of generating <br> tag in paragraph in range of 0-1.
	 * @param int    $min    Minimum number of sentences in paragraph.
	 * @param int    $max    Maximum number of sentences in paragraph.
	 * @param float  $image  Chance of generating image in paragraph in range of 0-1.
	 * @param float  $ul     Chance of generating <ul> tag in paragraph in range of 0-1.
	 * @param float  $ol     Chance of generating <ol> tag in paragraph in range of 0-1.
	 */
	public function __construct($imgAPI = null, $br = 0.1, $min = 5, $max = 12, $image = 0.1, $ul = 0.09, $ol = 0.09) {
		parent::__construct();
		$this->br = $br;
		$this->min = $min;
		$this->max = $max;
		$this->image = $image;
		$this->ul = $ul;
		$this->ol = $ol;
		$this->imgAPI = $imgAPI;
	}

	/**
	 * Generates an paragraph.
	 *
	 * @return string
	 */
	public function generateValue() {
		return $this->generateParagraph($this->imgAPI);
	}

	/**
	 * Generates HTML paragraph with random sentences.
	 *
	 * @param string $imgAPI API Key for pixabay.
	 *
	 * @return string Paragraph.
	 * @throws Exception
	 */
	public function generateParagraph($imgAPI = null) {

		$len = rand($this->min, $this->max);
		// Sometimes just return list
		foreach ([ 'ul', 'ol' ] as $tag) {
			if ($tag === 'ul') {
				if (static::should($this->ul)) {
					return $this->generateList($tag);
				}
			} elseif ($tag === 'ol') {
				if (static::should($this->ol)) {
					return $this->generateList($tag);
				}
			}
		}

		// Sometimes just return image
		if ($this->imgAPI !== null) {
			if (static::should($this->image)) {
				return $this->generateImage();
			}
		} elseif ($imgAPI !== null) {
			if (static::should($this->image)) {
				return $this->generateImage($imgAPI);
			}
		}

		// Generate paragraph
		$paragraph = '';
		for ($i = 0; $i < $len; $i++) {

			// Separate sentences
			$glue = '';
			if (strlen($paragraph)) {

				// Optional BR
				$useBR = static::should($this->br);
				$glue = $useBR ? '<br />' : ' ';
			}

			$paragraph .= $glue . parent::generateValue();
		}

		return parent::tag('p', $paragraph);
	}

	/**
	 * Generate a random number of paragraphs.
	 *
	 * @param string $imgAPI API Key for pixabay.
	 * @param int    $min    Minimum number of paragraphs.
	 * @param int    $max    Minimum number of paragraphs.
	 *
	 * @return string The joined paragraphs.
	 */
	public function generateParagraphs($imgAPI = null, $min = 3, $max = 5) {

		$len = rand($min, $max);

		$paragraphs = [ ];
		for ($i = 0; $i < $len; $i++) {
			$paragraphs[] = $this->generateParagraph($imgAPI);
		}

		return implode("\n\n", $paragraphs);
	}

	/**
	 * Generate an HTML list.
	 *
	 * @param string $tag The HTML tag to use, either 'ol' or 'ul'.
	 *
	 * @return string The HTML representation of a desired list.
	 */
	public function generateList($tag) {
		$len = rand($this->min, $this->max);

		$list = [ ];
		for ($i = 0; $i < $len; $i++) {
			$list[] = parent::tag('li', parent::generateValue());
		}

		return parent::tag($tag, implode("\n", $list));
	}

	/**
	 * Generates an HTML image.
	 *
	 * @param string $imgAPI API Key for pixabay.
	 * @param int    $width  Width of generated image.
	 * @param array  $params Specifies category of image.
	 *
	 * @return null|string
	 * @throws Exception
	 */
	public function generateImage($imgAPI = null, $width = 960, $params = [ 'travel' ]) {
		if ($this->imgAPI !== null) {
			$generator = new ImageGenerator($this->imgAPI, $width, $params);
			return '<img class="' . $this->htmlClass . '" src="' . $generator->generateValue() . '"/>';
		} elseif ($imgAPI !== null) {
			$generator = new ImageGenerator($imgAPI, $width, $params);
			return '<img class="' . $this->htmlClass . '" src="' . $generator->generateValue() . '"/>';
		} else {
			throw new Exception('Invalid API key.');
		}
	}

}